# Euler #2 in Elixir.

# Problem 2
# """
# Each new term in the Fibonacci sequence is generated by adding the 
# previous two terms. By starting with 1 and 2, the first 10 terms will be:

# 1, 2, 3, 5, 8, 13, 21, 34, 55, 89, ...

# Find the sum of all the even-valued terms in the sequence which do not 
# exceed four million.
# """

# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/

import Enum
import Euler
import NumberTheory

defmodule Euler2 do
  def euler2a() do
    1..100 |> map(fn n -> if rem(fib(n),2) == 0 && fib(n) < 4_000_000 do fib(n) else 0 end end) |> sum 
  end

  def e2b(n) do
    f = fib(n)
    if f < 4_000_000 && rem(f,2) == 0 do f else 0 end
  end
  
  def euler2b() do
    1..100 |> map(fn n -> e2b(n) end) |> sum 
  end
  
  def euler2c() do
    1..100 |> map(&fib/1) |> filter(fn n -> n < 4_000_000 && rem(n,2) == 0 end) |> sum 
  end

  # Recursive version (which is the proper way)
  def e2d(n,a) do
    f = fib(n)
    if f <= 4_000_000 do
      if rem(f,2) == 0 do
        e2d(n+1,f+a)
      else
        e2d(n+1,a)        
      end
    else
      a
    end
  end

  def euler2d() do
    e2d(1,0)
  end

  # Recursive version, just the fibs
  def e2e(n,a) do
    f = fib(n)
    if f <= 4_000_000 do
      e2e(n+1, [f | a])
    else
      a
    end
  end

  def euler2e() do
    e2e(1,[]) |> filter(&(rem(&1,2) == 0)) |> sum
  end

  # Using (lazy) Stream
  # Slower than the recursive ones (due to 2 calls to fib/1)
  def euler2f() do
    Stream.iterate(1, &(&1 + 1)) # generate numbers 1,2,3,4,5,....
    |> take_while(fn n -> fib(n) < 4_000_000 end) # this fib/1 makes it slower
    # |> IO.inspect    
    |> map(&(fib(&1))) # fib/1 again
    |> filter(&(rem(&1,2)==0))
    |> sum
  end

  # Same as euler2c/1 but with the proper range
  # But it's cheating!
  def euler2g() do
    1..33
    |> map(&fib/1)
    # |> filter(&{&1 <= 4_000_000 && rem(&1,2) == 0}) # Why does not this work?
    |> filter(fn n -> n <= 4_000_000 && rem(n,2) == 0 end)
    # |> IO.inspect
    |> sum 
  end

  # Same as euler2d but using the faster fib2/1
  def e2h(n,a) do
    f = fib2(n)
    if f <= 4_000_000 do
      if rem(f,2) == 0 do
        e2h(n+1,f+a)
      else
        e2h(n+1,a)        
      end
    else
      a
    end
  end

  def euler2h() do
    e2h(1,0)
  end

  
  def run_all() do
    # timeit(&Euler2.euler2a/0)
    # timeit(&Euler2.euler2b/0)
    # timeit(&Euler2.euler2c/0)
    # timeit(&Euler2.euler2d/0)
    # timeit(&Euler2.euler2e/0)
    # timeit(&Euler2.euler2f/0)
    # timeit(&Euler2.euler2g/0)
    timeit(&Euler2.euler2h/0)

  end

    def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler2.euler2a() end),
      # Task.async(fn -> Euler2.euler2b() end),
      # Task.async(fn -> Euler2.euler2c() end),
      # Task.async(fn -> Euler2.euler2d() end),
      # Task.async(fn -> Euler2.euler2e() end),
      # Task.async(fn -> Euler2.euler2f() end),
      # Task.async(fn -> Euler2.euler2g() end),
      Task.async(fn -> Euler2.euler2h() end),
    ]

    for task <- tasks do
      Task.await(task)
    end
    
  end

end
