#
# Euler #26 in Elixir.
#
# Problem 26
# """
# A unit fraction contains 1 in the numerator. The decimal representation of the 
# unit fractions with denominators 2 to 10 are given:
#
#     1/2	= 	0.5
#     1/3	= 	0.(3)
#     1/4	= 	0.25
#     1/5	= 	0.2
#     1/6	= 	0.1(6)
#     1/7	= 	0.(142857)
#     1/8	= 	0.125
#     1/9	= 	0.(1)
#     1/10	= 	0.1

# Where 0.1(6) means 0.166666..., and has a 1-digit recurring cycle. It can be 
# seen that 1/7 has a 6-digit recurring cycle.
#
# Find the value of d < 1000 for which 1/d contains the longest recurring cycle in 
# its decimal fraction part.
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler
import NumberTheory

defmodule Euler26 do

  #
  # Get the length of the repeating cycle for 1/n,
  # i.e. find the first instance where we got a repeated digit.
  #
  def get_rep_len(n) do
    1..n
    |> take_while(fn i -> pow_mod(10,i,n) != 1 end)
    |> length
    |> then(&(&1+1))
    end
  
  def euler26a() do
    {i,_n} = primes(999)
            |> map(fn i -> {i,get_rep_len(i)} end)
            |> max_by(fn {_i,n} -> n end) 
    i
  end

  def euler26b() do
    primes(999)
    |> map(fn i -> {i,get_rep_len(i)} end)
    |> max_by(fn {_i,n} -> n end) 
    |> elem(0)
  end

  def run_all() do
    # timeit(&Euler26.euler26a/0)
    timeit(&Euler26.euler26b/0)    
  end

  def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler26.euler26a() end),
      Task.async(fn -> Euler26.euler26b() end),      
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


