#
# Euler #27 in Elixir.
#
# Problem 27
# """
# Euler published the remarkable quadratic formula:
#
# n^2 + n + 41
#
# It turns out that the formula will produce 40 primes for the consecutive values 
# n = 0 to 39. However, when n = 40, 402 + 40 + 41 = 40(40 + 1) + 41 is divisible by 
# 41, and certainly when n = 41, 41^2 + 41 + 41 is clearly divisible by 41.
#
# Using computers, the incredible formula  n^2 − 79n + 1601 was discovered, which 
# produces 80 primes for the consecutive values n = 0 to 79. The product of the 
# coefficients, −79 and 1601, is −126479.
#
# Considering quadratics of the form:
#
#     n^2 + an + b, where |a| < 1000 and |b| < 1000
#
#     where |n| is the modulus/absolute value of n
#     e.g. |11| = 11 and |−4| = 4
#
# Find the product of the coefficients, a and b, for the quadratic 
# expression that produces the maximum number of primes for consecutive 
# values of n, starting with n = 0.
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler
import NumberTheory

defmodule Euler27 do

  # This is a port of my Curry/Haskell programs
  def p27(a, b) do
    n1 = 0
    pp = n1**2 + a*n1 + b
    if pp > 1 do p27_(pp,a,b,n1) else 0 end
  end

  def p27_(pp,a,b,n0) do
    n1 = n0 + 1
    pp1 = n1**2 + a*n1 + b
    if pp <= 1 || not prime?(pp) do
      n0
    else
      p27_(pp1,a,b,n1)
    end
  end
  
  # 0.90704s
  def euler27a() do
    t = 999
    for a <- -t..t, b <- -t..t do
      len = p27(a,b)
      {len,a*b,a,b}
    end
    |> max_by(fn {len,_ab,_a,_b} -> len end)
    |> elem(1)
  end

  def run_all() do
    timeit(&Euler27.euler27a/0)
  end

  def run_all_parallel() do
    tasks = [
      Task.async(fn -> Euler27.euler27a() end),
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


