defmodule Euler do
  # use Memoize
  import Enum
  
  @moduledoc """
  Documentation for `Euler`.

  This implements some Project Euler tasks (see https://projecteuler.net/ for more
  info about this).

  `@tasks` are for the fast tasks that are tested in `run_all/0` and `run_all_parallel/0`. 
  
  `@tasks_all` are _all_ the tasks that are tested in `run_alls/0` and 
  `run_all_parallel2/0`.   


  To run all tests from `mix`, run
      $ mix run_program
  or for specific task(s)
      $ mix run_program 10
      $ mix run_program 10 11 12

  """


  
  # The answers for the first 50 Project Euler problems:
  @answers %{
              Euler1=>233168,
              Euler2=>4613732,
              Euler3=>6857,
              Euler4=>906609,
              Euler5=>232792560,
              Euler6=>25164150,
              Euler7=>104743,
              Euler8=>40824,
              Euler9=>31875000,
              Euler10=>142913828922,
              Euler11=>70600674,
              Euler12=>76576500,
              Euler13=>5537376230,
              Euler14=>837799,
              Euler15=>137846528820,
              Euler16=>1366,
              Euler17=>21124,
              Euler18=>1074,
              Euler19=>171,
              Euler20=>648,
              Euler21=>31626,
              Euler22=>871198282,
              Euler23=>4179871,
              Euler24=>2783915460,
              Euler25=>4782,
              Euler26=>983,
              Euler27=>-59231,
              Euler28=>669171001,
              Euler29=>9183,
              Euler30=>443839,
              Euler31=>73682,
              Euler32=>45228,
              Euler33=>100,
              Euler34=>40730,
              Euler35=>55,
              Euler36=>872187,
              Euler37=>748317,
              Euler38=>932718654,
              Euler39=>840,
              Euler40=>210,
              Euler41=>7652413,
              Euler42=>162,
              Euler43=>16695334890,
              Euler44=>5482660,
              Euler45=>1533776805,
              Euler46=>5777,
              Euler47=>134043,
              Euler48=>9110846700,
              Euler49=>296962999629,
              Euler50=>997651,
    }

  
  # All the implemented Euler tasks
  @tasks [Euler1,Euler2,Euler3,Euler4,Euler5,Euler6,
          Euler7,Euler8,Euler9,
          Euler10, # slow (3.1s)
          Euler11,Euler12, Euler13,
          Euler14, # slow 7s
          Euler15,Euler16,
          Euler17,
          Euler18,Euler19,Euler20,Euler21,Euler22,
          Euler23, # a little too slow 2.37988s, 3.68349s
          Euler24, # too slow 2.76804s, 4.39142s 
          Euler25, # too slow 4.7879s 5.26743s
          Euler26,Euler27,Euler28,Euler29,Euler30,Euler31,Euler32,
          Euler33,Euler34,Euler35,Euler36,Euler37,Euler38,Euler39,
          Euler40,Euler41,Euler42,
          Euler43, # Too slow: 7.6s
          Euler44,Euler45,Euler46,Euler47,Euler48,Euler49,
          Euler50, # Too slow 2.50947s 2.45079s
             ]
  
  #
  # "System" functions.
  #
  
  
  @doc """
  timeit(fun)

  From https://stackoverflow.com/questions/29668635/how-can-we-easily-time-function-calls-in-elixir

  A more elaborate timing function. Prints
  * Name of the function
  * Result
  * Time in seconds

  ##Examples##

       > timeit(&Euler1.euler1a/0)
       &Euler1.euler1a/0 res:233168 time:0.00011s
       :ok

  """
  def timeit(fun) do
    # Convert the function (e.g. &Euler1.euler1a/0 to a string
    fun_s = Kernel.inspect(fun)
    {time0, res} = :timer.tc(fun, [])
    time = (time0 / 1_000_000) |> :erlang.float_to_binary([:compact, decimals: 5])
    # IO.puts("#{fun_s} res:#{res} time:#{time}s")
    [fun_s,res,time]
  end

  @doc """
  timeit_simple(fun) 

  A simple timing function, returns the time in seconds (as a string).

  ##Examples##

       > Euler.timeit_simple(&Euler.run_all/0)

       > Euler.timeit_simple(fn Euler1.run_all() end)

  """
  def timeit_simple(fun) do
    {time0, res} = :timer.tc(fun, [])
    IO.inspect(res)
    (time0 / 1_000_000) |> :erlang.float_to_binary([:compact, decimals: 5]) 
  end

  
  @doc """
  run_all() 
  
  Runs all the Euler tasks in `@tasks`.
  """
  def run_all() do
    times = %{}
    [_ok,not_ok, times,total_time] = run_all(@tasks,[],[], times,0)
    IO.inspect(total_time, label: "total_time")
    IO.inspect(sort_by(times, fn {s,_} -> Regex.replace(~r{.*&?Euler(\d+).*},s,"\\1") |> String.to_integer end), label: "times sort on key", limit: :infinity)
    IO.inspect(sort_by(times, fn {_k,v} -> v |> String.to_float end, :desc), label: "times sort on times desc", limit: :infinity)
    IO.inspect(not_ok |> reverse, label: "not_ok")    
  end

  @doc """
  run_all(task) 

  Runs a single task (`task`).

  This is used in task/run_program.

  ##Examples##

      > mix run_program 14

      > mis run_program 14 23
  """
  def run_all(task) do
    run_all([task],[],[],%{},0) 
  end
  
  defp run_all([],ok,not_ok,times,total_time), do: [ok,not_ok,times,total_time]
  defp run_all([task | tasks],ok,not_ok,times,total_time) do
    [fun_s,res,time] = task.run_all()
    # fun = Regex.replace(~r{.*&?Euler(\d+).*},fun_s,"Euler\\1")
    if @answers[task] == res do 
      IO.inspect([fun_s,res,time,:ok])
      run_all(tasks,
              [fun_s | ok],
              not_ok,
              Map.put(times, fun_s, time),total_time+String.to_float(time))
    else
      IO.inspect([fun_s,res,time,:not_ok])      
      run_all(tasks,
              ok,
              [fun_s | not_ok],
              Map.put(times, fun_s, time),total_time+String.to_float(time))
    end   
  end
  
  @doc """
  run_all_parallel() 
  
  Runs all the Euler tasks in `@tasks` "in parallel".

  ##Examples##

       > Euler.run_all_parallel()

  """  
  def run_all_parallel() do
    # This does not work well
    # for e <- @tasks do
    #   # IO.inspect(e, label: "problem")
    #   # Euler.timeit_simple(fn -> Task.await(Task.async(&e.run_all_parallel/0),:infinity) end)
    #   # |> IO.inspect
    
    # Running all 1..50 takes 12.3s 
    @tasks
    |> Enum.map(fn e -> Task.async(fn -> e.run_all end) end)
    |> Task.await_many(:infinity)
    # |> IO.inspect(limit: :infinity)
    |> show_res

  end

  # 
  # Note: This is much of a duplicate of run_all/1 which is not good.
  # TODO:
  #  - Fix show show_res/1 can be used for both run_all/1 and show_res/1?
  #
  defp show_res(_answers, [],ok,not_ok,times,total_time), do: [ok,not_ok,times,total_time]
  defp show_res(answers, [ [fun_s,res,time] | tasks],ok,not_ok,times,total_time) do
    fun = Regex.replace(~r{.*&?Euler(\d+).*},fun_s,"Euler\\1")
    if Map.fetch!(answers,fun) == res do 
      IO.inspect([fun_s,res,time,:ok])
      show_res(answers,
               tasks,
               [fun_s | ok],
               not_ok,
               Map.put(times, fun_s, time),total_time+String.to_float(time))
    else
      IO.inspect([fun_s,res,time,:not_ok])      
      show_res(answers,
               tasks,
               ok,
               [fun_s | not_ok],
               Map.put(times, fun_s, time),total_time+String.to_float(time))
    end   
  end
  
  defp show_res(res) do
    # Convert to string representation of the @answer (instead of Module names)
    answers = @answers |> map(fn {t,res} -> {t |> to_string |> String.trim("Elixir."), res} end) |> Enum.into(%{})
    times = %{}
    [_ok,not_ok, times,total_time] = show_res(answers,res,[],[],times,0)
    IO.inspect(total_time, label: "total_time")
    IO.inspect(sort_by(times, fn {_k,v} -> v |> String.to_float end, :desc ), label: "times sort on times desc", limit: :infinity)
    IO.inspect(not_ok |> reverse, label: "not_ok")    
  end
  
    
  #
  # Misc functions
  #  

  @doc """
  palindrome?(s) 

  Returns true if `s` is a palindrome, otherwise false.


  ##Examples##
  
       iex> palindrome?("1234" |> String.to_charlist)
       false

       iex> palindrome?("1234321" |> String.to_charlist)
       true

       iex> palindrome?(1234321 |> Integer.digits)
       true

  """
  def palindrome?(s) do
    s == Enum.reverse(s)
  end

  @doc """
  transpose(m) 

  Returns a transposed version of `m`.

  ##Example##

       iex> [[1,2,3],[4,5,6],[7,8,9]] |> transpose
       [[1, 4, 7], [2, 5, 8], [3, 6, 9]]
  """
  def transpose(m) do
    Enum.zip_with(m, &Function.identity/1)
  end

  @doc """
  mat_at(m,i,j)

  Returns the value (`i`,`j`) of the 2d matrix `mat`.

  ##Examples##

       iex> [[1,2,3],[4,5,6],[7,8,9]] |> mat_at(1,2)
       6
  """
  def mat_at(m,i,j) do
    m |> Enum.at(i) |> Enum.at(j)
  end
  
  @doc """
  format_num(num, prec \\ 2)
  
  Format a number to `prec' (default 2) chars.

  ##Examples##

      iex> format_num(3)
      "03"

      iex> format_num(3,3)
      "003"

      iex> format_num(13)
      "13"
  
  """
  def format_num(num, prec \\ 2), do: Integer.to_string(num) |> String.pad_leading(prec,"0")
  

  @doc """
  but_last(list)

  Returns all elements in `list` except the last.

  ##Examples##

      iex> but_last([1,2,3,4,5])
      [1,2,3,4]

      iex> but_last([])
      []
  """
  def but_last([]), do: []
  def but_last(list) do
    Enum.drop(list,-1) # This seems to be the fastest
    # Enum.take(list,(list |> length) - 1) # quite fast
    
    # Testing other approaches from
    # https://stackoverflow.com/questions/52319984/remove-last-element-from-list-in-elixir)   
    # list |> Enum.reverse() |> tl() |> Enum.reverse() # about the same
    # [k,_] = Enum.chunk_every(list, length(list)-1)  # slower
    # k
    # {_, k} = List.pop_at(list, -1) # slower
    # k
    # List.delete_at(list, length(list)-1) # about the same
    # list |> List.to_tuple() |> Tuple.delete_at(length(list)-1) |> Tuple.to_list # about the same
  end

  @doc """
  permutation(list) 

  Returns all permutations of `list`.

  From https://elixirforum.com/t/most-elegant-way-to-generate-all-permutations/2706

  ##Examples##

      iex> permutations([1,2,3])
      [[1, 2, 3], [1, 3, 2], [2, 1, 3], [2, 3, 1], [3, 1, 2], [3, 2, 1]]

  """
  def permutations([]), do: [[]]
  def permutations(list) do
    for elem <- list, rest <- permutations(list--[elem]), do: [elem|rest]
  end

  @doc """
  is_permutation?(a,b) 

  Returns true of b is a permutation of a, otherwise returns false.

  ##Examples##
  
      iex> is_permutation?([1,2,3],[3,1,2])
      true
  """
  def is_permutation?(a,b) do
    (a |> Enum.sort) == (b |> Enum.sort)
  end

  
  @doc """
  rotate(list,i) 

  Returns a list where the `lst` is rotated 'i` steps to the right.

  ##Examples##
      iex> rotate(1..10,4)
      [5, 6, 7, 8, 9, 10, 1, 2, 3, 4]
  """
  def rotate(lst,i) do
    Enum.drop(lst,i) ++ Enum.take(lst,i)
  end

  @doc """
  rotate_all(list) 

  Returns a list of all (right) rotations of `lst`.

  ##Examples##
      iex> rotate_all(1..4 |> Enum.to_list)
      [[1, 2, 3, 4], [2, 3, 4, 1], [3, 4, 1, 2], [4, 1, 2, 3]]
  """
  def rotate_all(lst) do
    for i <- 0..length(lst)-1 do
      rotate(lst,i)                           
    end
  end
  

end
