#
# Euler #14 in Elixir.
#
# Problem 14
# """
# The following iterative sequence is defined for the set of positive integers:
#
# n n/2 (n is even)
# n 3n + 1 (n is odd)
#
# Using the rule above and starting with 13, we generate the following 
# sequence:
#     13 40 20 10 5 16 8 4 2 1
#
# It can be seen that this sequence (starting at 13 and finishing at 1) 
# contains 
# 10 terms. Although it has not been proved yet (Collatz Problem), it is 
# thought that all starting numbers finish at 1.
#
# Which starting number, under one million, produces the longest chain?
#
# NOTE: Once the chain starts the terms are allowed to go above one million.)
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler

defmodule Euler14 do

  # def collatz(1), do: 1  
  def collatz(n) when rem(n,2) == 0 do
    div(n,2)
  end
  def collatz(n) when rem(n,2) == 1  do
    3*n + 1
  end

  # def collatz(1), do: 1
  # def collatz(n) do
  #   if rem(n,2) == 0 else 3*n + 1 end
  # end
  
  defp collatzm(1,memo) do
    memo
  end
  defp collatzm(n,memo) do
    c = memo[n]
    if c == nil do
      c2 = collatz(n)
      memo2 = Map.put(memo, n, c2)
      collatzm(c2,memo2)
    else
      collatzm(c,memo)
    end
  end

  # Another take
  defp collatzm2(1,memo,len) do
    {len,memo}
  end
  defp collatzm2(n,memo,len) do
    c = memo[n]
    if c == nil do
      c2 = collatz(n)
      memo2 = Map.put(memo, n, c2)
      collatzm2(c2,memo2,len+1)
    else
      collatzm2(c,memo,len+1)
    end
  end

  
  # Returns the list of collatz numbers. 
  def collatz_list(1) do
    []
  end
  def collatz_list(n) when n > 1 do
    c = collatz(n)
    # [c] ++ collatz_list(c)  # Not tail-recursive
    [c | collatz_list(c)]
  end

  # Returns the list of collatz numbers.
  # Using accumulator (tail-recursive). Faster.
  def collatz_list2(1,lst) do
    lst # faster for this application
    # lst |> reverse # fo collatz_list2(c,[c|lst])  in collatz_list2/2, seems to be slower
  end
  def collatz_list2(n,lst) when n > 1 do
    c = collatz(n)
    collatz_list2(c,[c] ++ lst) # faster
    # collatz_list2(c,[c|lst]) # a little slower 
    # collatz_list2(c,lst ++ [c]) # much slower
  end

  # Accumulator is the length.
  # Not correct!
  # def collatz_list3(1,len) do
  #   len
  # end
  # def collatz_list3(n,len)  do
  #   c = collatz(n)
  #   collatz_list3(c,len + 1) 
  # end

  #
  # Too slow (unsurprisingly): 42.26173s
  # However, it does not memoize.
  #
  def euler14a() do
    memo = %{}
    2..1_000_000 
    |> map(fn n -> {n,collatzm(n,memo)} end)
    |> map(fn {n, k} -> {n,Map.keys(k) |> length} end)
    |> IO.inspect
    |> max_by(fn {_n,k} -> k end)
    |> IO.inspect
  end

  #
  # Using collatz_list/1 is faster: 14.99117s
  # Still too slow, though.
  #
  def euler14b() do
    {n,_len} = 2..1_000_000 
              |> map(fn n -> {n,collatz_list(n)} end)
              |> map(fn {n, k} -> {n,length(k)} end)    
              |> max_by(fn {_n,len} -> len end)
    n
    |> IO.inspect
  end

  # Helper function fir euler13c/0
  def e14c(n,memo,max_val,max_len) do
    if n < 1_000_000 do
      {len,memo2} = collatzm2(n,memo,0)
      if len > 500 do
        {max_val,max_len}
      end
      if len > max_len do
        e14c(n+1,memo2,n,len)
      else
        e14c(n+1,memo2,max_val,max_len)        
      end
    else
      {max_val,max_len}
    end
  end

  # Recursive variant with accumulators: 14.88528s
  def euler14c() do
    memo = %{}
    e14c(2,memo,2,0)
  end

  # Better: 6.97241s 6.95804s 7.39796s
  def euler14d() do
    2..1_000_000 
    |> map(fn n -> {n,length(collatz_list2(n,[n]))} end)
    |> max_by(fn {_n,len} -> len end)
    |> elem(0)
  end

  
  def run_all() do
    # timeit(&Euler14.euler14a/0)
    # timeit(&Euler14.euler14b/0)
    # timeit(&Euler14.euler14c/0)
    timeit(&Euler14.euler14d/0)
  end

  def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler14.euler14a() end),
      # Task.async(fn -> Euler14.euler14b() end),
      # Task.async(fn -> Euler14.euler14c() end),
      Task.async(fn -> Euler14.euler14d() end),
    ]

    for task <- tasks do
      Task.async(task)
    end
  end
  
end


