#
# Euler #25 in Elixir.
#
# Problem 25
# """
# The Fibonacci sequence is defined by the recurrence relation:
#
#    Fn = Fn1 + Fn2, where F1 = 1 and F2 = 1.
#
# Hence the first 12 terms will be:
#
#    F1 = 1
#    F2 = 1
#    F3 = 2
#    F4 = 3
#    F5 = 5
#    F6 = 8
#    F7 = 13
#    F8 = 21
#    F9 = 34
#    F10 = 55
#    F11 = 89
#    F12 = 144
#
# The 12th term, F12, is the first term to contain three digits.
#
# What is the first term in the Fibonacci sequence to contain 1000 digits?")
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#
import Enum
import Euler
import NumberTheory
  
defmodule Euler25 do

  # Too slow: 4.7879s 5.26743s
  def euler25a() do
    ( Stream.iterate(1,&(&1+1))
    |> take_while(fn n -> length(Integer.digits(fib(n))) < 1000 end)
    |> length
    ) + 1
  end

  # Using then/1 instead of (... ) + 1
  def euler25b() do
    Stream.iterate(1,&(&1+1))
    |> take_while(fn n -> length(Integer.digits(fib(n))) < 1000 end)
    |> length
    |> then(&(&1+1))
  end

  # Recursive variant: not faster
  def e25c(n) do
    if length(Integer.digits(fib(n))) >= 1000 do
      n
      else
      e25c(n+1)
    end    
  end
  def euler25c() do
    e25c(1)
  end

  # From euler25c but using the faster fib2/1
  # And it's faster: 1.59488s 1.72617
  def e25d(n) do
    # Using fib3/1 is about the same time
    if length(Integer.digits(fib2(n))) >= 1000 do
      n
      else
      e25d(n+1)
    end    
  end
  def euler25d() do
    e25d(1)
  end

  
  def run_all() do
    # timeit(&Euler25.euler25a/0)
    # timeit(&Euler25.euler25b/0)
    # timeit(&Euler25.euler25c/0)
    timeit(&Euler25.euler25d/0)
  end

  def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler25.euler25a() end),
      # Task.async(fn -> Euler25.euler25b() end),
      # Task.async(fn -> Euler25.euler25c() end),
      Task.async(fn -> Euler25.euler25d() end),
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


