#
# Euler #35 in Elixir.
#
# Problem 35
# """
# The number, 197, is called a circular prime because all rotations 
# of the digits: 197, 971, and 719, are themselves prime.
#
# There are thirteen such primes below 100: 
# 2, 3, 5, 7, 11, 13, 17, 31, 37, 71, 73, 79, and 97.
#
# How many circular primes are there below one million? 
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler
import NumberTheory

defmodule Euler35 do

  def rotate_all_primes?(n) do
    rotate_all(n |> Integer.digits)
    |> all?(fn lst -> prime?(lst |> Integer.undigits) end )
  end
  
  # A little too slow: 2.07177s
  def euler35a() do
    for n <- 1..1_000_000, prime?(n), rotate_all_primes?(n) do
      n
    end
    |> count
  end


  # Helper for euler35b/0
  def e35b(n) do
    lst = n |> Integer.digits
    # We know that n is a prime
    for i <- 1..length(lst)-1 do
      prime?(rotate(lst,i) |> Integer.undigits)
    end
    |> all?()
  end
  
  # A little more hard coded to the problem, and a little faster: 2.06759s
  def euler35b() do
    for n <- 1..1_000_000, prime?(n), e35b(n) do
      n
    end
    |> count
    
  end

  # Not faster: 2.10713s
  def euler35c() do
    filter(1..1_000_000,fn n -> prime?(n) && e35b(n) end)
    |> count
    
  end
  
  def run_all() do
    # timeit(&Euler35.euler35a/0)
    timeit(&Euler35.euler35b/0)
  end

  def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler35.euler35a() end),
      Task.async(fn -> Euler35.euler35b() end),       
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


