#
# Euler #43 in Elixir.
#
# Problem 43
# """
# The number, 1406357289, is a 0 to 9 pandigital number because it is made up of 
# each of the digits 0 to 9 in some order, but it also has a rather interesting 
# sub-string divisibility property.
#
# Let d1 be the 1st digit, d2 be the 2nd digit, and so on. In this way, we 
# note the following:
#
#     * d2d3d4=406 is divisible by 2
#     * d3d4d5=063 is divisible by 3
#     * d4d5d6=635 is divisible by 5
#     * d5d6d7=357 is divisible by 7
#     * d6d7d8=572 is divisible by 11
#     * d7d8d9=728 is divisible by 13
#     * d8d9d10=289 is divisible by 17
#
# Find the sum of all 0 to 9 pandigital numbers with this property.
# """
#
# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler
# import NumberTheory

defmodule Euler43 do
        
  # Using lists
  def check_perm_a(p,t) do    
    for i <- 0..6, v = 100*at(p,i+1) + 10*at(p,i+2) + at(p,i+3), rem(v,at(t,i)) == 0  do
      p
    end
  end

  # Too slow: 11.5s
  def euler43a() do
    t = [2,3,5,7,11,13,17]
    ps = 0..9 |> to_list
    for p <- permutations(ps),
             at(p,0) > 0,
             length(check_perm_a(p,t)) == 7 do
        # p
        Integer.undigits(p)
    end
    # |> map(&Integer.undigits/1)
    |> sum
  end


  # Using tuples instead
  def check_perm_b(p,t) do
    # for i <- 0..6, v = 100*elem(p,i+1) + 10*elem(p,i+2) + elem(p,i+3), rem(v,elem(t,i)) == 0  do
    # This not (discernible) faster
    for i <- 0..6, rem(100*elem(p,i+1) + 10*elem(p,i+2) + elem(p,i+3),elem(t,i)) == 0  do
       p
    end
  end

  # Using tuples instead
  # Still too slow: 7.49744s 7.55358s 7.34074s
  def euler43b() do
    t = {2,3,5,7,11,13,17}
    ps = 0..9 |> to_list
    for p <- permutations(ps) |> map(fn p -> List.to_tuple(p) end),
       elem(p,0) > 0,
       length(check_perm_b(p,t)) == 7 do
      # p
      Integer.undigits(p |> Tuple.to_list)
    end
    # |> map(fn p -> Integer.undigits(p |> Tuple.to_list) end)
    |> sum
  end


  # euler43c/0 is an improved version, suggested by Boris Okner (private communication)
  defp permutations2(list, criteria \\ nil)
  defp permutations2([], _criteria), do: [[]]
  defp permutations2(list, criteria) do
    for elem <- list, rest <- permutations2(list--[elem]), !criteria || criteria.(elem, rest) do
      [elem | rest]
    end
  end
  
  def check_perm_c(p,t) do
      Enum.all?(0..6, fn i ->
        v = 100*at(p,i+1) + 10*at(p,i+2) + at(p,i+3)
        rem(v, elem(t,i)) == 0
      end)
  end

  # This is faster: 4.5s-5.5s
  def euler43c() do
    t = {2,3,5,7,11,13,17}
    ps = 0..9 |> to_list
    permutations2(ps, fn el, rest ->
      (length(rest) < 9) || (el > 0 && check_perm_c([el | rest],t))
    end)
    |> map(fn p -> Integer.undigits(p) end)
    |> sum
  end


  
  def run_all() do
    # timeit(&Euler43.euler43a/0)
    # timeit(&Euler43.euler43b/0)
    timeit(&Euler43.euler43c/0)    
  end

  def run_all_parallel() do
    tasks = [
      # Task.async(fn -> Euler43.euler43a() end),
      # Task.async(fn -> Euler43.euler43b() end),
      Task.async(fn -> Euler43.euler43c() end),      
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


