#
# Euler #43 in Elixir.
#
# Problem 43
# """
# The number, 1406357289, is a 0 to 9 pandigital number because it is made up of 
# each of the digits 0 to 9 in some order, but it also has a rather interesting 
# sub-string divisibility property.
#
# Let d1 be the 1st digit, d2 be the 2nd digit, and so on. In this way, we 
# note the following:
#
#     * d2d3d4=406 is divisible by 2
#     * d3d4d5=063 is divisible by 3
#     * d4d5d6=635 is divisible by 5
#     * d5d6d7=357 is divisible by 7
#     * d6d7d8=572 is divisible by 11
#     * d7d8d9=728 is divisible by 13
#     * d8d9d10=289 is divisible by 17
#
# Find the sum of all 0 to 9 pandigital numbers with this property.
# """
#
# This is Boris Okner's version. Alas, it's about 1s slower than euler43b/0
# in euler43.ex
# 

# This Elixir program was created by Hakan Kjellerstrand, hakank@gmail.com
# See also my Elixir page: http://www.hakank.org/elixir/
#

import Enum
import Euler
# import NumberTheory

defmodule Euler43BorisOkner do

  # From Boris Okner
  def permutations_boris(list, criteria \\ nil)
  def permutations_boris([], _criteria), do: [[]]
  def permutations_boris(list, criteria) do
    for elem <- list, rest <- permutations_boris(list--[elem]), !criteria || criteria.(elem, rest) do
      [elem | rest]
    end
  end

  # def check_perm_c(p,t) do
  #   for i <- 0..6, v = 100*at(p,i+1) + 10*at(p,i+2) + at(p,i+3), rem(v, elem(t,i)) == 0  do
  #      p
  #   end
  # end

  # # Using tuples instead
  # def euler43c() do
  #   t = {2,3,5,7,11,13,17}
  #   ps = 0..9 |> to_list
  #   permutations_boris(ps, fn el, rest ->
  #     el > 0 &&
  #     length(check_perm_c([el | rest],t)) == 7
  #   end)
  #   |> map(fn p -> Integer.undigits(p) end)
  #   |> sum
  # end

  # Second take: This is faster: 4.5s (vs 7.2s)!
  def check_perm_c(p,t) do
      Enum.all?(0..6, fn i ->
        v = 100*at(p,i+1) + 10*at(p,i+2) + at(p,i+3)
        rem(v, elem(t,i)) == 0
      end)
  end

  # Using tuples instead
  # 
  def euler43c() do
    t = {2,3,5,7,11,13,17}
    ps = 0..9 |> to_list
    permutations_boris(ps, fn el, rest ->
      (length(rest) < 9) || (el > 0 && check_perm_c([el | rest],t))
    end)
    |> map(fn p -> Integer.undigits(p) end)
    |> sum
  end
  
  def run_all() do
    timeit(&Euler43BorisOkner.euler43c/0)    
  end

  def run_all_parallel() do
    tasks = [
      Task.async(fn -> Euler43BorisOkner.euler43c() end),
    ]

    for task <- tasks do
      Task.await(task)
    end
  end
  
end


