#
# Number theoretic utilities for Project Euler
#
defmodule NumberTheory do
  use Memoize
  import Enum
  
  @doc """
  prime?(n)

  Returns true is `n` is a prime, otherwise false.

  **Examples**

      iex> prime?(2**100-1)
      false

  """
  # def prime?(2) do
  #   true
  # end
  # def prime?(3) do
  #   true
  # end
  # def prime?(4) do
  #   false
  # end
  # def prime?(n) do
  #   # 3..(div(n,2)+1)
  #   # [2] ++ to_list(3..div(n,2)+1)
  #   # |> all?(fn i -> rem(n,i) != 0 end)
  #   prime_(n,2)
  # end

  # # Helper function for prime?/1
  # defp prime_(n,2) do
  #   if rem(n,2) == 0 do
  #     false
  #   else
  #     prime_(n,3)
  #   end
  # end
  # defp prime_(n,i) do
  #   if i > div(n,2) do
  #     true
  #   else
  #     if rem(n,i) == 0 do
  #       false
  #     else
  #       prime_(n,i+2)
  #     end
  #   end
  # end

  ## From https://gist.github.com/thebugcatcher/2487b9ed7f70ed39aa4afec86c730665
  ## 
  # def prime?(n) when n in [2, 3], do: true
  # def prime?(n) do
  #   floored_sqrt = :math.sqrt(n)
  #   |> Float.floor
  #   |> round
  #   !Enum.any?(2..floored_sqrt, &(rem(n, &1) == 0))
  # end
  # This is faster
  def prime?(1), do: false  
  def prime?(n) when n in [2,3,5,7], do: true
  def prime?(n) when rem(n,2) == 0, do: false
  def prime?(n) when rem(n,3) == 0, do: false
  def prime?(n) when rem(n,5) == 0, do: false
  def prime?(n) when rem(n,7) == 0, do: false
  def prime?(n) when n >= 11 do
    !Enum.any?(2..round(Float.floor(:math.sqrt(n))), &(rem(n, &1) == 0))
  end

  @doc """
  prime_memo?(n)

  Returns true is `n` is a prime, otherwise false.
  Memoized version.

  Alas, this seems to be quite slower than using prime?/1. Why?

  **Examples**

      iex> prime_memo?(2**100-1)
      false

  """
  ## def prime_memo?(n), do: prime?(n)  # For comparison
  defmemo prime_memo?(n), do: prime?(n)

  
  @doc """
  prime_factors(n):
  Returns all prime factors of `n`.

  **Examples**
  
      iex> prime_factors(2**100-1)
      [3, 5, 5, 5, 11, 31, 41, 101, 251, 601, 1801, 4051, 8101, 268501]

  """
  def prime_factors(1), do: []
  def prime_factors(n) do
    if prime?(n) do
      [n]
    else
      fs = prime_factors(n,2,[]) |> Enum.reverse()
      fs != [n] && fs || []
    end
  end
  def prime_factors(n,2,a) do
    if rem(n,2) == 0 do
      prime_factors(div(n,2),2,[2 | a])
    else
      prime_factors(n,3,a)
    end
  end
  def prime_factors(n,i,a) do
    if i <= Float.ceil(:math.sqrt(n)+1) do
      if prime?(i) do
        if rem(n,i) == 0 do
          prime_factors(div(n,i),i,[i | a])
        else
          prime_factors(n,i+2,a)      
        end
      else
        prime_factors(n,i+2,a)      
      end
    else
      if n > 1 do
        [n | a] 
      else
        a
      end
    end
  end

  @doc """
  prime_factors2(n):
  Returns all prime factors of `n`.

  This is an alternative version of prime_factors/1 using "plain" recursion instead
  of using an accumulator.

  **Examples**
  
       iex> prime_factors2(2**100-1)
       [3, 5, 5, 5, 11, 31, 41, 101, 251, 601, 1801, 4051, 8101, 268501]
  """
  # Another version
  def prime_factors2(n) do
    fs = prime_factors2(n,2) 
    fs != [n]  && (fs |> Enum.reverse()) || []
    
  end
  # Check 2
  def prime_factors2(n,2) do
    if rem(n,2) == 0 do
      prime_factors2(div(n,2),2) ++ [2]
    else
      prime_factors2(n,3)
    end
  end
  # Check 3..2..sqrt(n)
  def prime_factors2(n,i) do
    if i <= Float.ceil(:math.sqrt(n)+1) do
      if prime?(i) do
        if rem(n,i) == 0 do
          prime_factors2(div(n,i),i) ++ [i]
        else
          prime_factors2(n,i+2)      
        end
      else
        prime_factors2(n,i+2)      
      end
    else
      if n > 1 do
        [n] 
      else
        []
      end
    end
  end

  @doc """
  prime_divisors(n)

  Returns the prime divisors of `n`.

  ##Examples##

       iex> prime_divisors(2**10-1)
       [3, 11, 31]

  """
  def prime_divisors(n) do
    # 2..ceil(:math.sqrt(n))
    # |> filter(fn i -> rem(n,i) == 0 && prime?(i) end)
    prime_factors(n) |> uniq
  end

  @doc """
  next_prime(n)

  Returns the next prime of `n`.

  ##Examples##

       iex> next_prime(8)
       11

       iex> next_prime(10)
       11

       iex> next_prime(11)
       13
  """
  def next_prime(2) do
    3
  end
  def next_prime(n) do
    next_prime_(n,n+1)
  end
  
  defp next_prime_(n,i) do
    if prime?(i) do
      i
    else
      next_prime_(n,i+1)        
    end
  end

  @doc """
  nprimes(count)

  Returns the `count` first primes.

  Note: This is slow for larger values of `count`.

  ##Examples##

       iex> nprimes(10)
       [2, 3, 5, 7, 11, 13, 17, 19, 23, 29]
  """
  def nprimes(limit) do
    nprimes(limit,1,3,[2]) 
  end
  defp nprimes(limit,c,i,a) do
    if c < limit do
      if prime?(i) do
        nprimes(limit,c+1,i+2,[i | a])
      else
        nprimes(limit,c,i+2,a)
      end
    else
      a |> reverse
    end
  end
  
  ## Using next_prime/1 is slower
  # def nprimes(limit) do
  #   nprimes(limit,1,3,[2]) 
  # end
  # defp nprimes(limit,c,i,a) do
  #   if c < limit do
  #     nprimes(limit,c+1,next_prime(i),[i | a])
  #   else
  #     a |> reverse
  #   end
  # end

  
  @doc """
  primes(limit)

  Returns the primes <= `limit`.

  Note: This is slow for larger values of `limit`.
  
  ##Examples##

       iex> primes(20)
       [2, 3, 5, 7, 11, 13, 17, 19]
  """
  ## From https://gist.github.com/thebugcatcher/2487b9ed7f70ed39aa4afec86c730665
  def primes(n) when n < 2, do: []
  def primes(n), do: Enum.filter(2..n, &prime?(&1))
    
  # def primes(limit) do
  #   primes(limit,3,[2]) 
  # end
  # defp primes(limit,i,a) do
  #   if i < limit do
  #     if prime?(i) do
  #       primes(limit,i+2,[i | a])
  #     else
  #       primes(limit,i+2,a)
  #     end
  #   else
  #     a |> reverse
  #   end
  # end

  @doc """
  primes2(limit)

  A variant of `primes/1` using `next_prime/1` instead.
  However, its is slower than `primes/1`.

       Euler.timeit_simple(fn -> primes(100_000) end)
       "2.15825"
       Euler.timeit_simple(fn -> primes2(100_000) end)
       "4.30316"

  ##Examples##

       iex> primes2(20)
       [2, 3, 5, 7, 11, 13, 17, 19]  
  """
  def primes2(limit) do
    primes2(limit,3,[2]) 
  end
  defp primes2(limit,i,a) do
    if i < limit do
      if prime?(i) do
        primes2(limit,next_prime(i),[i | a])
      else
        primes2(limit,next_prime(i),a)
      end
    else
      a |> reverse
    end
  end
 
  @doc """
  divisors(n)

  Returns all divisors of `n`.

  ##Examples##

       iex> divisors(28)
       [1, 2, 4, 7, 14, 28]

  """
  def divisors(n) do
    half = if n <= 2 do n else div(n,2) end # Fix division by 0
    [1] ++
    for i <- 2..half, rem(n,i) == 0 do
      i
    end ++ [n]
    |> uniq
  end

  @doc """
  num_divisors(n)

  Returns the number of divisors of `n`.

  ##Examples##

       iex> num_divisors(28)
       6
  """
  def num_divisors(n) do
    prime_factors(n) |> frequencies |> Map.values |> map(&(&1 + 1)) |> product    
  end


  @doc """
  proper_divisors(n)

  Returns the proper divisors of `n`, i.e. `divisors` except `n` itself.

  ##Examples##

      iex> proper_divisors(100)
      [1, 2, 4, 5, 10, 20, 25, 50]

      iex> proper_divisors(1)
      [1]

      iex> proper_divisors(2)
      [1]
  """
  def proper_divisors(1), do: [1]
  def proper_divisors(n) do
    half = if n <= 2 do n else div(n,2) end # Fix division by 0
    [1] ++
    for i <- 2..half, rem(n,i) == 0 do
      i
    end
    |> uniq
    |> List.delete(n)
  end


  @doc """
  proper_divisors_sum(n)

  Returns the sum of the proper divisors of `n`.

  ##Examples##

      iex> proper_divisors_sum(1)
      1
      iex> proper_divisors_sum(2)
      1

      iex> 1..10 |> Enum.map(fn n -> proper_divisors_sum(n) end)
      [1, 1, 1, 3, 1, 6, 1, 7, 4, 8]
  """
  def proper_divisors_sum(1), do: 1
  def proper_divisors_sum(n) do
    proper_divisors(n) |> sum
  end

  

  @doc """
  lcm(x,y)

  Calculates the lcm of x and y

  ##Examples##

       iex> lcm(123,32) 
       3936

       iex> Enum.reduce(2..20,1,&lcm/2) 
       232792560

  """
  def lcm(x,y) do
    g = Integer.gcd(x,y)
    div(x*y,g)
  end

  @doc """
  Fibonacci

  fib/1: The call to fibm/2 using an empty Map

  ##Examples###

       iex> fib(10)
       55

       iex> fib(1000)
       43466557686937456435688527675040625802564660517371780402481729089536555417949051890403879840079255169295922593080322634775209689623239873322471161642996440906533187938298969649928516003704476137795166849228875
  """
  def fib(n) do
    {val, _memo} = fibm(n, %{})
    val
  end

  @doc """
  Fibonacci

  fibm/2: The memoize function

  ##Examples##

       iex> fibm(10,%{})
       {55,
       %{2 => 1, 3 => 2, 4 => 3, 5 => 5, 6 => 8, 7 => 13, 8 => 21, 9 => 34, 10 => 55}}

  """
  def fibm(0, memo), do: {0, memo}
  def fibm(1, memo), do: {1, memo}
  def fibm(n, memo) do
    val = memo[n]
    if val == nil do
      {f1, memo1} = fibm(n - 1, memo)
      {f2, memo2} = fibm(n - 2, memo1)
      n2 = f1 + f2
      {n2, Map.put(memo2, n, n2)}
    else
      {val, memo}
    end
  end
  
  #
  # Using Memoize.
  # Nice, but it seems to be a little slower than fib/1
  # timeit(fn () ->  fib(100_000) end): 1.26683s
  # timeit(fn () ->  fibs(100_000) end): 1.59714s
  # The second time is faster:
  # fib/1: 0.66815
  # fibs/1: 0.00003 But this use the existing cache which is not what I want
  # Ah, one can use
  #   Memoize.invalidate(Euler, :fibs)
  # to clear the cache. Great! Though it's still slower than fib/1.  :-)
  ### defmemo fibs(0), do: 0
  ### defmemo fibs(1), do: 1
  ### defmemo fibs(n), do: fibs(n - 1) + fibs(n - 2)
  #
  # See https://stackoverflow.com/questions/29668635/how-can-we-easily-time-function-calls-in-elixir
  # def measure(function) do
  #   function
  #   |> :timer.tc
  #   |> elem(0)
  #   |> Kernel./(1_000_000) 
  # end

  @doc """
  fib2(n)

  A tail recursive version of Fibonacci.
  This is a little faster than fib/1 (which use memoization).

  Comparison:
  - fib/1:
    > :timer.tc(fn -> fib(100000) end) |> elem(0)
    678413


  - fib2/1:
    > :timer.tc(fn -> fib2(100_000) end) |> elem(0)
    238839
    (i.e. 0.238839s

    > :timer.tc(fn -> fib2(1_000_000) end) |> elem(0)
    -> Cannot allocate 16925484064 bytes of memory

    > :timer.tc(fn -> fib2(1_000_000) end) |> elem(0)
    11472600
    (i.e. 11.472600s)

    This is a quite large string
    > fib2(1_000_000) |> Integer.to_string |> String.length
    208988


  ##Examples##

       iex> fib2(10)
       55

  """
  def fib2(n) when n < 0, do: :error
  def fib2(n), do: fib2(n,1,0)
  def fib2(0,_, res), do: res
  def fib2(n,next,res), do: fib2(n-1, next + res, next)
  

  @doc """
  fib3(n)

  Another tail recursive version of Fibonacci.
  This is a little faster than fib/1 (which use memoization).

  However, it's not significantly faster than fib2/0.

  ## Examples ##
       iex> fib3(10)
       55

       iex> fib(100)
       354224848179261915075

  """
  def fib3(n), do: fib3(n,1,1,1)
  defp fib3(n,n,a,_b), do: a
  defp fib3(n,i,a,b), do: fib3(n,i+1,b,a+b)
  

  @doc """
  factorial(n)

  Returns the factorial of `n`

  ##Examples##
  
      iex> factorial(3)
      6

      iex> factorial(10)
      3628800

      iex> factorial(30)
      265252859812191058636308480000000
  """
  def factorial(0), do: 1  
  def factorial(1), do: 1
  def factorial(n), do: 2..n |> reduce(&(&1 * &2))
  
  @doc """
  pow_mod(n,p,m)

  Returns the value of n**p mod m.

  ##Examples##

      iex> pow_mod(23,100,35)  # 23**100 mod 35
      16
  """
  def pow_mod(n,p,m), do: rem(n**p,m)

  @doc """
  nlen(n)

  Returns the length of a decimal number.

  ##Examples##

      iex> nlen(23)
      2

      iex> nlen(748317)
      6

      iex> nlen(2**100-1)
      31
  """
  def nlen(n), do: 1+:math.floor(:math.log10(n)) |> trunc
 
  
end
